/**
 * \file: exchndd.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * Exception handler daemon.
 *
 * \component: exchnd
 *
 * \author: Kai Tomerius (ktomerius@de.adit-jv.com)
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/epoll.h>

#include <systemd/sd-daemon.h>

#include "exchnd_backend.h"
#include "exchnd_interface.h"
#include "exchnd_collector.h"
#include "exchndd.h"


/* ---------------------------------------------------------------------- */
/* command line interface */

/* USAGE - help text for the command line interface */
#ifdef DLT
#   define USAGE_DLT                                       \
    "    -u               write exceptions also to DLT\n"
#else /* ifdef DLT */
#   define USAGE_DLT
#endif /* ifdef DLT */

#define USAGE_SERVER                                                    \
    "    --signal-server=<client_app>\n"                                \
    "                          Notify the application <client_app>\n"   \
    "                          when a signal is received by an\n"       \
    "                          application. <client_app> must be\n"     \
    "                          in /usr/bin. The notification is done\n" \
    "                          the same way systemd would do it with\n" \
    "                          ExecStopPost=<client_app> guilty_app\n"  \
    "                          service option. Signal information\n"    \
    "                          is then given through environment.\n"

#if (EXCHND_VERSION > 0x206)
#   define USAGE_ABORT                                              \
    "    --4k-stack-on-abort   The daemon will provide 4K of\n"     \
    "                          stack dump in case of SIGABRT\n"     \
    "    --ext-bt-on-abort     Extended backtrace on SIGABRT\n"     \
    "                          The daemon tries to retrieve\n"      \
    "                          arguments passed to functions.\n"    \
    "                          Up to 10 potential arguments.\n"     \
    "    --ext-bt              Extended backtrace.\n"               \
    "                          The daemon tries to retrieve\n"      \
    "                          arguments passed to functions.\n"    \
    "                          Up to 10 potential arguments.\n"     \
    "    --frame-bt-fallback   Fall back to frame based\n"          \
    "                          backtrace if the exidx backtrace\n"  \
    "                          gave unexpected last function.\n"    \
    "    --guess-fallback=[size] The daemon run through the stack\n" \
    "                          and look for valid function\n"       \
    "                          pointers, then dump their name\n"    \
    "                          and position. The research is\n"     \
    "                          limited to [size] bytes of stack.\n" \
    "                          Search in the complete used stack\n" \
    "                          if [size] is negative.\n"            \
    "                          Found function names are guessed\n"  \
    "                          and may be wrong.\n"                 \
    "    --ext-bt-fallback     Fall back to extended backtrace.\n"  \
    "                          The daemon tries to retrieve\n"      \
    "                          arguments passed to functions\n"     \
    "                          if the backtrace failed.\n"          \
    "                          Up to 10 potential arguments.\n\n"

#else
#   define USAGE_ABORT
#endif

#define USAGE                                                     \
    "usage: exchndd -D [-d <device>] [-c <path>]\n\n"             \
                                                                  \
    "usage: exchndd --smoketest [-d <device>]\n\n"                \
                                                                  \
    "    -d, --device <device> select exception handler device\n" \
    "    -D, --demonize        run as daemon\n\n"                 \
                                                                  \
    "    -T, --terminate       exit after last command\n\n"       \
                                                                  \
    "    -K, --kill            Kill the client process once\n"    \
    "                          data collection is done\n"         \
    "    -S, --shutdown        Shutdown the system once\n"        \
    "                          data collection is done\n\n"       \
    USAGE_ABORT                                                   \
    "    -f, --file <path>     print exceptions on <path>\n"      \
    USAGE_DLT                                                     \
                                                                  \
    "    --max-bt-depth=depth  Max backtrace depth (def: 32)\n"   \
    "    -s, --smoketest       test device and exit.\n\n"         \
    "    -m, --mem             memory pool size in bytes"         \
    " (default 1MB).\n\n"                                         \
    USAGE_SERVER                                                  \

#define MAX_EPOLL_EVENTS 20

static int memory_pool = MEMORY_POOL;

/* usage - command line help */
static void usage()
{
    fprintf(stderr, USAGE);
}

long action = EXCHND_NO_ACTION;
int guess_size = 512;
int max_bt_depth = MAX_BACKTRACE_DEPTH;

static char *dev_name = "/dev/exchnd";

#define FLEN sizeof("/proc/xxxxx/cmdline")

static void get_process_name_by_pid(const int pid, char name[32])
{
    char file_name[FLEN] = { '\0' };

    if (name) {
        FILE *file = NULL;

        snprintf(file_name, FLEN, "/proc/%d/comm", pid);

        file = fopen(file_name, "r");

        if (file) {
            size_t size;

            size = fread(name, sizeof(char), 16, file);

            if (size > 0)
                name[size - 1] = '\0';

            name[16] = '\0';

            fclose(file);
        }
    }

    return;
}

/* catch_signal - signal handler */
static void catch_signal(int signo, siginfo_t *si, void *ucontext)
{
    char name[32] = { '\0' };
    (void)ucontext;

    get_process_name_by_pid(si->si_pid, name);

    switch (signo) {
    case SIGINT:
    case SIGQUIT:
    case SIGTERM:
        exchnd_print_error("Signal %d received from %s (%d), exiting.",
                           signo,
                           name,
                           si->si_pid);
        action |= EXCHND_TERMINATE;
        exchnd_handle_event();
        break;
    case SIGHUP:
        /* Let's get some backtrace */
        show_backtrace();
        break;
    case SIGSEGV:
    default:
        /* Let's get some backtrace */
        show_backtrace();
        exchnd_print_error("Daemon killed with %d.", signo);
        exit(-1);
        break;
    }
}

/* install_signal_handler - install a handler for some signals */
static void install_signal_handler()
{
    int signals[] = { SIGINT, SIGQUIT, SIGTERM, SIGSEGV, SIGHUP, 0 };
    unsigned int i;
    struct sigaction sa;

    /* install a signal handler for the above listed signals */
    for (i = 0; signals[i]; i++) {
        memset(&sa, 0, sizeof(sa));
        sa.sa_sigaction = catch_signal;
        sa.sa_flags |= SA_SIGINFO;

        if (sigaction(signals[i], &sa, NULL) < 0)
            exchnd_print_error(
                "Failed to install signal %u handler. Error: %s\n",
                signals[i], strerror(errno));
    }
}

/* ---------------------------------------------------------------------- */

int parse_arguments(int argc, char **argv)
{
    int c;
    int parse = 1;
    static int temp = 0;

    while (parse) {
        static struct option long_options[] = {
            { "smoketest", no_argument, &temp, EXCHND_SMOKETEST },
            { "help", no_argument, NULL, 'h' },
            { "terminate", no_argument, &temp, EXCHND_TERMINATE },
            { "kill", no_argument, &temp, EXCHND_KILL },
            { "shutdown", no_argument, &temp, EXCHND_SHUTDOWN },
            { "demonize", no_argument, &temp, EXCHND_DEMONIZE },
#if (EXCHND_VERSION > 0x206)
            { "4k-stack-on-abort", no_argument, &temp, EXCHND_EXT_STACK },
            { "ext-bt-on-abort", no_argument, &temp, EXCHND_EXT_BT },
#endif
            { "ext-bt", no_argument, &temp, EXCHND_EXT_BT_ALL },
            { "ext-bt-fallback", no_argument, &temp, EXCHND_EXT_BT_FALLBACK },
            { "frame-bt-fallback", no_argument, &temp,
              EXCHND_FRAME_BT_FALLBACK },
            { "guess-fallback", required_argument, &temp,
              EXCHND_GUESS_FALLBACK },
            { "trace", no_argument, NULL, 't' },
            { "dlt", no_argument, NULL, 'u' },
            { "file", required_argument, NULL, 'f' },
            { "mem", required_argument, NULL, 'm' },
            { "device", required_argument, NULL, 'd' },
            { "signal-server", required_argument, &temp, EXCHND_SERVER },
            { "max-bt-depth", required_argument, &temp, EXCHND_MAX_BT_DEPTH },
            { 0, 0, 0, 0 }
        };

        c = getopt_long(argc,
                        argv,
                        "hDTd:o:f:tusm:KS",
                        long_options,
                        NULL);

        if (c == -1)
            break;

        switch (c) {
        case 0:
            action |= temp;

            if (temp == EXCHND_GUESS_FALLBACK)
                guess_size = strtol(optarg, NULL, 10);

            if (temp == EXCHND_SERVER)
                set_client_name(optarg);

            if (temp == EXCHND_MAX_BT_DEPTH)
                max_bt_depth = strtol(optarg, NULL, 10);

            break;

        case 'd':
            dev_name = optarg;
            break;

        case 'o':
        /* Backward compatibility */
        case 'f':
            backend_mask |= FILE_BACKEND;
            exchnd_set_file_name(optarg);
            break;

        case 'u':
            backend_mask |= DLT_BACKEND;
            break;

        case 't':
            backend_mask |= TRACE_BACKEND;
            break;

        case 's':
            action |= EXCHND_SMOKETEST;
            break;
        case 'T':
            action |= EXCHND_TERMINATE;
            break;
        case 'D':
            action |= EXCHND_DEMONIZE;
            break;
        case 'h':
            action = EXCHND_HELP;
            break;
        case 'S':
            action |= EXCHND_SHUTDOWN;
            break;
        case 'K':
            action |= EXCHND_KILL;
            break;
        case 'm':

            if (isdigit(optarg[0]))
                memory_pool = atoi(optarg);

            break;

        default:
            /* Unrecognized parameter. */
            action = EXCHND_HELP;
            break;
        }
    }

    return 0;
}

/* ---------------------------------------------------------------------- */
int main(int argc, char **argv)
{
    ExchndInterface_t *exh_if = NULL;
    struct epoll_event exh_ev;
    int exh_epfd;

    int ret = EXIT_FAILURE;

    parse_arguments(argc, argv);

#ifndef __arm__

    if (action & EXCHND_FRAME_BT_FALLBACK)
        /* This fall back has no use on other platform than arm */
        action &= ~EXCHND_FRAME_BT_FALLBACK;

#endif

    /* handle various command line options */
    if (action & EXCHND_HELP) {
        /* help */
        usage();
        return EXIT_SUCCESS;
    }

    if (action & EXCHND_SMOKETEST) {
        /* smoke-test */
        exh_if = exchnd_create(dev_name);
        exchnd_info(exh_if, 2);
        return exh_if ? EXIT_SUCCESS : ret;
    }

    if (action & EXCHND_TERMINATE)
        /* not run as a daemon -> end */
        return EXIT_SUCCESS;

    if (action & EXCHND_DEMONIZE) {
        /* daemonize */
        if (daemon(0, 0)) {
            exchnd_print_error("Failed to daemonize");
            return EXIT_FAILURE;
        }

        /* log the process id in /tmp/exchndd.pid */
        FILE *f = fopen("/tmp/exchndd.pid", "w");

        if (f) {
            fprintf(f, "%u\n", getpid());
            fclose(f);
        }
    }

    /* Now we are in daemon mode */

    /* create the exception handler interface */
    exh_if = exchnd_create(dev_name);

    if (exh_if == NULL) {
        exchnd_print_error("Failed to create exception hander interface");
        unlink("/tmp/exchndd.pid");
        goto unlink_exchnd;
    }

    /* Prepare data structures for poll call */
    exh_epfd = epoll_create(1);

    if (exh_epfd < 0) {
        exchnd_print_error("exchnd epoll_create failed. Err=%s",
                           strerror(errno));
        goto destroy_interface;
    }

    exh_ev.events = EPOLLIN;
    exh_ev.data.fd = exh_if->efd;

    if (epoll_ctl(exh_epfd, EPOLL_CTL_ADD, exh_if->efd, &exh_ev) == -1)
        goto destroy_epoll;

    /* From now on we catch exceptions that will be handled later. */
    if (!(action & EXCHND_DEMONIZE))
        sd_notify(0, "READY=1");

    /* Loading arch specific library. */
    if (!libunwind_load())
        goto destroy_epoll;

    /* handle some signals */
    install_signal_handler();

    /* Set highest priority so that we can handle exception faster. */
    exchnd_init_scheduling();

    if ((action & EXCHND_SERVER) && exchnd_init_server()) {
        exchnd_print_error("Unable to initialize server: %s", strerror(errno));
        goto unwind_unload;
    }

    if (exchnd_init_storage()) {
        exchnd_print_error("Unable to initialize storage: %s", strerror(errno));
        goto server_unload;
    }

    if (action & (EXCHND_EXT_BT | EXCHND_EXT_STACK | EXCHND_EXT_BT_ALL))
        if (exchnd_get_version(exh_if) < 0x207) {
            action &= ~(EXCHND_EXT_BT | EXCHND_EXT_STACK | EXCHND_EXT_BT_ALL);
            exchnd_print_error("4k stack on abort and extended backtrace "
                               "on abort are not supported with this driver. "
                               "Please update to v2.07 or superior.");
        }

    if (exchnd_lock_memory(memory_pool)) {
        exchnd_print_error("Unable to lock memory.");
        goto msg_cleanup;
    }

    if (exchnd_init_processing(exh_if)) {
        exchnd_print_error("Unable to initialize processing");
        action = EXCHND_TERMINATE;
        goto unlock_mem;
    }

    /* We don't need future lock anymore */
    exchnd_mem_rellock();

    /* Main loop */
    /* From now we can handle exception that may have been caught earlier. */
    while (1) {
        int n = 0;
        int nevents = 0;
        struct epoll_event events[MAX_EPOLL_EVENTS];

        if (action & EXCHND_TERMINATE)
            /* not run as a daemon -> end */
            break;

        nevents = epoll_wait(exh_epfd, events, MAX_EPOLL_EVENTS, 500);

        if (nevents < 0) {
            if (errno == EINTR) {
                /* Only exit if the daemon has received QUIT/INT/TERM */
                ret = EXIT_SUCCESS;
                continue;
            }

            exchnd_print_error("exchnd device poll failed. Err=%s",
                               strerror(errno));

            ret = EXIT_FAILURE;
            break;
        }

        for (n = 0; n < nevents; n++) {
            if (!(events[n].events & EPOLLIN)) {
                exchnd_print_error("exchnd device poll returned"
                                   " revents=0x%x.errno=%s",
                                   events[nevents].events,
                                   strerror(errno));
                break;
            }

            if (exchnd_handle_event()) {
                action = EXCHND_TERMINATE;
                ret = EXIT_FAILURE;
                break;
            }
        }

        /* If rc == 0 just continue */
    }

    /* Ensure action is in the correct state. */
    action = EXCHND_TERMINATE;
    exchnd_deinit_processing();

unlock_mem:
    exchnd_unlock_memory();

msg_cleanup:
    /* clean up and terminate */
    exchnd_cleanup_storage();

server_unload:

    /* Destroying the server */
    if (action & EXCHND_SERVER)
        exchnd_cleanup_server();

unwind_unload:
    /* Closing libunwind */
    libunwind_unload();

destroy_epoll:
    /* Closing epoll file descriptor */
    close(exh_epfd);

destroy_interface:
    exchnd_destroy(exh_if);

unlink_exchnd:
    unlink("/tmp/exchndd.pid");
    exchnd_print_error("Exception handler daemon exited");

    return ret;
}